using Combinatorics
using CPUTime
using Dates, Printf, Statistics
using SQLite, DBInterface, DataFrames

using PyCall
struct my_wrapper_pysat
    formula
    solvers
    card
    pb
end
# If no super user rights available just:
# git clone https://github.com/pysathq/pysat.git
# cd pysat
# pip install .
@pyimport pysat.formula as pysat_dot_formula #import pysat.formula
@pyimport pysat.solvers as pysat_dot_solvers #import pysat.solvers
@pyimport pysat.card as pysat_dot_card #import pysat.card
@pyimport pysat.pb as pysat_dot_pb #import pysat.pb    # requires in python: `pip install pypblib`
pysat = my_wrapper_pysat(pysat_dot_formula, pysat_dot_solvers, pysat_dot_card, pysat_dot_pb)

@pyimport threading as py_threading # for interrupting a solver-run started via pysat

function as_prettyfied_string_permutationslist(plist, linenumbers=false)
    return "[\n" * join(linenumbers ? enumerate(plist) : plist, ",\n") * "\n]"
end


"""
Assuming earlier we've e.g. done:
`forconstellation_prepare__experiment_series("~/prestorage_subgroups.sqlite", "~/experiments_collection.sqlite", 1, 6, 4, 3, 3, 2, "glucose3", 3600)`
E.g., my_cmdstring = "SELECT * FROM constellations WHERE meant_for_leftcoset_approach=1 AND d=6 AND n = 4 AND k = 3 AND kappa = 3 and subgrouporder=2"

`steer_experim_viaSQL("~/experiments_collection.sqlite", "SELECT * FROM constellations WHERE meant_for_leftcoset_approach=1 AND d=12 AND n = 4 AND k = 4 AND kappa = 3 and (subgrouporder=12 OR subgrouporder=6)", 1)`


"""
function steer_experim_viaSQL(db_filename, my_cmdstring, limit_iter=1000000, write_to_sql=true)
    ourlimit = 0
    while true
        sleep(0.016) # ATTENTION, as we do not want to setup an SQL server, let's just be pragmatic by giving a bit of time to close connection, `SQLiteException("database is locked")`
        db = SQLite.DB(db_filename)
        CMDSTR = my_cmdstring * " AND access_status!='being_processed' AND termination_status_experiment!='already_examined' LIMIT 1"
        df = DBInterface.execute(db, CMDSTR) |> DataFrame
        SQLite.close(db)

        if isempty(df) || ourlimit > limit_iter
            println("We stopped because in Database no not-yet examined configuration is available, or true==$(ourlimit > limit_iter), the maximum number of experiments has been reached.")
            break
        end
        ourlimit += 1
        # In case we want to run this in parallel:
        sleep(0.016) # ATTENTION, as previous comment
        db = SQLite.DB(db_filename)

        df2 = DBInterface.execute(db, "UPDATE constellations SET access_status='being_processed' WHERE d=$(df[!,:d][1]) AND n=$(df[!,:n][1]) AND k=$(df[!,:k][1]) AND kappa=$(df[!,:kappa][1]) AND subgrouporder=$(df[!,:subgrouporder][1]) AND subgroup_as_tuplelist='$(df[!,:subgroup_as_tuplelist][1])' AND timelimit_given=$(df[!,:timelimit_given][1])") |> DataFrame
        SQLite.close(db)

        CONFIG = Dict([
            "meant_for_leftcoset_approach" => Bool(df[!, :meant_for_leftcoset_approach][1]),
            "d" => df[!, :d][1],
            "n" => df[!, :n][1],
            "k" => df[!, :k][1],
            "kappa" => df[!, :kappa][1],
            "subgroup_as_tuplelist" => eval(Meta.parse(df[!, :subgroup_as_tuplelist][1])),
            "solver_employed" => df[!, :solver_employed][1],
            "card_constraint_mode" => df[!, :card_constraint_mode][1],
            "lex_strength" => df[!, :lex_strength][1],
            "timelimit_given" => df[!, :timelimit_given][1]
        ])

        if write_to_sql == true
            offset_permutations::Vector{Vector{Int}}, pdesign_obtained::Vector{Vector{Int}}, feasibility, solver_finished_in_time, solvertime_measured, julias_tic_measured, solver_report = carryout_search(CONFIG)

            db = SQLite.DB(db_filename)
            my_loc_cmd = "UPDATE constellations SET access_status='currently_unprocessed',termination_status_experiment='already_examined', offset_permutations='$(isempty(offset_permutations) ? "[]" : offset_permutations)', pdesign_obtained='$(isempty(pdesign_obtained) ? "[]" : pdesign_obtained)', feasibility_check_passed=$(Int(feasibility)), solver_finished_in_time=$(Int(solver_finished_in_time)), solvertime_measured= $solvertime_measured, julias_tic_measured=$julias_tic_measured, solver_report='$solver_report' WHERE d=$(df[!,:d][1]) AND n=$(df[!,:n][1]) AND k=$(df[!,:k][1]) AND kappa=$(df[!,:kappa][1]) AND subgrouporder=$(df[!,:subgrouporder][1]) AND subgroup_as_tuplelist='$(df[!,:subgroup_as_tuplelist][1])' AND timelimit_given=$(df[!,:timelimit_given][1])"

            @show(my_loc_cmd)

            df2 = DBInterface.execute(db, my_loc_cmd) |> DataFrame
            SQLite.close(db)
        else
            carryout_search(CONFIG)
        end
    end
end

function solving_an_xor(d, n, k)
    XOR_CLAUSES = []
    Combinatorics.Permutations(1:n, kappa)
    for l in 1:d
        for my_set in combinations(1:n, k)
            push!(XOR_CLAUSES, [X[(l, my_perm)] for my_perm in Combinatorics.Permutations(myset, length(myset))])
        end
    end
end

function gen_SOP_with_cardinalities_range(card_range, n)
    SOP = []
    for j in card_range
        act_coll = []
        for act_comb_j in combinations(1:n, j)
            for idx in 1:length(act_comb_j)
                copy_act_comb_j = copy(act_comb_j)
                el = act_comb_j[idx]
                deleteat!(copy_act_comb_j, idx)
                push!(act_coll, tuple([el; sort(copy_act_comb_j)]...))
            end
        end
        sort!(act_coll) # here, sorting is just a matter of taste
        append!(SOP, act_coll)
    end
    return SOP
end

"""
E.g., for
`
[4, 5, 1, 2, 6, 3], [1, 6, 5, 3, 4, 2], [5, 4, 6, 2, 1, 3], [6, 1, 4, 3, 5, 2], [6, 2, 4, 5, 3, 1], [4, 3, 2, 1, 6, 5], [2, 6, 3, 5, 4, 1], [3, 4, 6, 1, 2, 5], [5, 3, 2, 6, 1, 4], [2, 1, 3, 4, 5, 6], [3, 5, 1, 6, 2, 4], [1, 2, 5, 4, 3, 6]]
`
"""
function is_rkw_indep(pmat, k)
    d = length(pmat)
    n = length(pmat[1])

    if maximum(pmat[1]) < n
        pmat = (x -> (y -> y + 1).(x)).(pmat) # (Apparently we are in the setting of alphabets {0,...,n-1}. Hence, bring to {1,..., n}.)
    end

    mydict_of_not_met = []
    global_counter = 0
    is_valid = true
    for subperm in Combinatorics.Permutations{Vector{Integer}}(1:n, k)
        counter = 0
        for l in 1:length(pmat)
            if all([pmat[l][subperm[j-1]] < pmat[l][subperm[j]] for j in 2:length(subperm)])
                counter += 1
                global_counter += 1
            end
        end

        if counter != div(d, factorial(k))
            push!(mydict_of_not_met, (subperm, counter))
            is_valid = false
        end
    end

    @show(global_counter == d * binomial(n, k))
    is_valid == false && println("mydict_of_not_met for subperms = ", mydict_of_not_met)
    return is_valid
end

"""
Checks if the list of permutations `pmat' is `k'-restricted minwise independent.
E.g., for
`
A = [[4, 5, 1, 2, 6, 3],
    [1, 6, 5, 3, 4, 2],
    [5, 4, 6, 2, 1, 3],
    [6, 1, 4, 3, 5, 2],
    [6, 2, 4, 5, 3, 1],
    [4, 3, 2, 1, 6, 5],
    [2, 6, 3, 5, 4, 1],
    [3, 4, 6, 1, 2, 5],
    [5, 3, 2, 6, 1, 4],
    [2, 1, 3, 4, 5, 6],
    [3, 5, 1, 6, 2, 4],
    [1, 2, 5, 4, 3, 6]
]
is_mw_indep(A, 4)
'
returns `true'.
"""
function is_mw_indep(pmat, k)
    @assert k >= 2

    SOP = gen_SOP_with_cardinalities_range(2:k, length(pmat[1]))
    mydict_of_not_met = []
    is_valid = true
    global_counter = 0
    for semiseq in SOP
        counter = 0
        for l in 1:length(pmat)
            if all([pmat[l][semiseq[1]] < pmat[l][semiseq[j]] for j in 2:length(semiseq)])
                counter += 1
                global_counter += 1
            end
        end

        if counter != div(length(pmat), length(semiseq))
            push!(mydict_of_not_met, (semiseq, counter))
            is_valid = false
        end
    end

    @show (global_counter, sum(length(pmat) * binomial(length(pmat[1]), kappa) for kappa in 2:k))
    is_valid == false && println("mydict_of_not_met = ", mydict_of_not_met)
    return is_valid
end

function perminverting_function(perm)
    inv_perm = [0 for i in 1:length(perm)]
    for i in 1:length(perm)
        inv_perm[perm[i]] = i
    end
    return inv_perm
end

function apply_perm(outer, inner)
    @assert length(outer) == length(inner) "Permutation-concatenation works just for common symmetric group on n elements!"
    return [outer[inner[j]] for j in 1:length(inner)]
end

"""
Given a permutation `perm`, the function returns the binary matrix x_{i,j}, i,j=1, ..., n,
with x_{i,j} = 1 iff `perm[i] < perm[j]`.
"""
function permutation_to_incidencematrix(perm)
    # Initialize the incidence matrix with zeros
    incidence_matrix = [0 for i in 1:length(perm), j in 1:length(perm)]

    # Fill in the incidence matrix according to the total order
    for i in 1:length(perm)
        for j in i+1:length(perm)
            if perm[i] < perm[j]
                incidence_matrix[i, j] = 1
            else
                incidence_matrix[j, i] = 1
            end
        end
    end
    return incidence_matrix
end

function permutationmatrix_to_permutation(mat)
    return [sum([mat[i, j] * j for j in 1:size(mat)[1]]) for i in 1:size(mat)[1]]
end

function cnf2ilp!(target_ilpmodel, variables_dict, list_of_clauses)
    # TODO Future work
    for clause in list_of_clauses
        #@constraint(target_ilpmodel, sum(literal > 0 ? variables_dict[literal] : 1 - variables_dict[(-1)*literal] for literal in clause) >= 1)
    end
end

function traverse_sup_diagonals_linearly(my_matrix)
    index_traversal = []
    for s in 2:size(my_matrix)[1]
        for t in 0:size(my_matrix)[1]-s
            push!(index_traversal, my_matrix[t+1, s+t])
        end
    end
    return index_traversal
end

"""
Function returning a formula in CNF guaranteeing that the vectors of
Boolean variables `P` is a lexicographic predecessor of `Q`,
i.e., answering if
`P <==lex== Q'
holds true.
I.e., if the vectors are interpreted as the coefficients of
decreasing powers of two for the binary number representation,
then the respective first binary number is smaller than the second one.
E.g., if
`
julia> P = [0,1,0,0,1]
julia> Q = [0,1,1,0,0]
',
then the formula will contain (P, Q) as satisfying assignment.

TODO length = n or length n-1 ?? CHECKTHIS!

Here `P' and `Q' are vectors of integers, where each entry points to a literal.

The parameter `X' asks for a length-(n-1) list of fresh Boolean variables,
where n = min(length(P), length(Q)); these are used as axuliary variables
in the formulation of
"Section 3.2: The AND Encoding using Common Subexpression Elimination"
https://digitalcommons.iwu.edu/cgi/viewcontent.cgi?article=1022&context=cs_honproj

in non-available: Elgabou et al. 2015, https://www.researchgate.net/publication/292427792_Encoding_the_lexicographic_ordering_constraint_in_sat_modulo_theories ,  
Encoding the lexicographic ordering constraint in satisfiability modulo theories,
apparently at least available as thesis: https://etheses.whiterose.ac.uk/10387/
that will be returned as CNF.
"""
function lex_and_ordering(P, Q, X)
    if length(P) == 0 || length(Q) == 0
        return []
    end

    M = P[1:minimum([length(P), length(Q)])]
    N = Q[1:minimum([length(P), length(Q)])]

    additional_clauses = []

    push!(additional_clauses, [N[1], (-1) * M[1]]) # (Add N[1] >= M[1], equivalently: N[1] || -M[1] .)

    # From this previous line the condition for X[1] <-> (N[1]==M[1]) is also clear:
    # Case ``X[1]'':
    append!(additional_clauses,
        [
            #[X[i], (-1)*X[i+1]], # (In this particular case a tautology.)
            #[N[i+1], (-1)*N[i+1], (-1)*X[i+1]], # (Removed tautology.)
            [(-1) * X[1], N[1], (-1) * M[1]], # (Interpreted as X[1] -> (N[1] >= M[1]) .)
            [(-1) * X[1], M[1], (-1) * N[1]], # (Interpreted as X[1] -> (M[1] >= N[1]) .)
            #[M[i+1], (-1)*M[i+1], (-1)*X[i+1]], # (Removed tautology.)
            [X[1], (-1) * N[1], (-1) * M[1]],  # (Interpreted as -X[1] -> ( min(N[1],M[1])==0) .)
            [X[1], N[1], M[1]] # (Interpreted as -X[1] -> (max(N[1],M[1])==1) .)
        ])

    for i in 1:length(N)-2
        # Encode formula X[i+1] <-> (X[i] && N[i+1] == M[i+1]) : 

        append!(
            additional_clauses,
            [
                [X[i], (-1) * X[i+1]],
                #[N[i+1], (-1)*N[i+1], (-1)*X[i+1]], # (Removed tautology.)
                [(-1) * X[i+1], N[i+1], (-1) * M[i+1]], # (Interpreted as X[i+1] -> (N[i+1] <= M[i+1]) .)
                [(-1) * X[i+1], M[i+1], (-1) * N[i+1]], # (Interpreted as X[i+1] -> (M[i+1] >= N[i+1]) .)
                #[M[i+1], (-1)*M[i+1], (-1)*X[i+1]], # (Removed tautology.)
                [X[i+1], (-1) * N[i+1], (-1) * M[i+1], (-1) * X[i]], # (Interpreted as -X[i+1] -> (X[i] -> min(N[i+1], M[i+1]) == 0) .)
                [X[i+1], N[i+1], M[i+1], (-1) * X[i]] # (Interpreted as -X[i+1] -> (X[i] -> max(N[i+1], M[i+1]) == 1) .)
            ]
        )
    end


    for i in 1:length(N)-1
        push!(additional_clauses, [(-1) * X[i], N[i+1], (-1) * M[i+1]]) # (Interpreted as X[i] -> (N[i+1] || -M[i+1]), i.e., X[i] -> (N[i+1] >= M[i+1]) .)
    end

    return additional_clauses
end

function runtests_lexorder_check()
    passed_tests = []

    for L in 1:8
        M = collect(1:L)
        N = collect(L+1:2*L)
        X = collect(2*L+1:3*L)

        result = lex_and_ordering(M, N, X)

        my_starting_clauses = PyObject([PyObject([PyObject(el) for el in row]) for row in result])

        @pywith pysat.solvers.Solver(name="glucose421", bootstrap_with=my_starting_clauses, use_timer=true) as solver begin
            count_of_solutions = 0
            while solver.solve() #&& count_of_solutions < 5
                interpretation_found = solver.get_model()
                @show interpretation_found
                count_of_solutions += 1

                bigvee = []
                vec1 = []
                vec2 = []
                for lit in M
                    push!(bigvee, (-1) * Int(interpretation_found[abs(lit)]))
                    push!(vec1, Int(interpretation_found[abs(lit)] >= 1))
                end
                for lit in N
                    push!(bigvee, (-1) * Int(interpretation_found[abs(lit)]))
                    push!(vec2, Int(interpretation_found[abs(lit)] >= 1))
                end

                @show vec1
                @show vec2
                is_lexicographically_ordered = (vec1 == vec2 || sortperm([vec1, vec2]) == [1, 2])
                @assert is_lexicographically_ordered "This should never happen and would certify incorrectness of the modeling approach."

                solver.add_clause(bigvee)
            end
            @show (div(2^L * (2^L + 1), 2), count_of_solutions)
            @assert (div(2^L * (2^L + 1), 2)) == count_of_solutions "Modeling approach apparently does not find all solutions." # (The count of lexicographical pairs must be equal to Gaussian summation term.)
            (div(2^L * (2^L + 1), 2)) == count_of_solutions && push!(passed_tests, L)
        end
    end
    @show passed_tests
end

"""
    The following function captures symbolically the truth value of a given CNF and "stores" it within the fresh variable `z_boolean_result_keeper` by 
    enforcing the validity of z_boolean_result_keeper <-> CNF, where this latter formula is itself returned in CNF.
    `z_result'
    `y_subexpression_result'
    """
function symbolic_evaluation_of_cnf(CNF, z_boolean_result_keeper, y_auxiliary_variables_disjunctions_substitutions)

    if length(CNF) == 0 # even needed, or already handled by more general case?
        # An empty CNF, the conjunction over an empty set of expressions, is evaluated to true, hence,  we
        # return a unit clause guaranteeing `z_boolean_result_keeper=1'
        return [[z_boolean_result_keeper]]
    end

    additional_clauses = []

    for idx in 1:length(CNF)
        # The following "stores" the truth value of each clause inside the respective y-entry:

        # Add the constraint y[idx] <-> (CNF[idx][1] || CNF[idx][2] || ... || CNF[idx][end]),
        # firstly, by adding y[idx] -> (CNF[idx][1] || CNF[idx][2] || ... || CNF[idx][end]),
        push!(additional_clauses, vcat([(-1) * y_auxiliary_variables_disjunctions_substitutions[idx]], [CNF[idx][l] for l in 1:length(CNF[idx])]))
        #####println("Adding for clauseidx=$idx the clause 'y[idx] -> (CNF[idx][1] || CNF[idx][2] || ... || CNF[idx][end])'")
        #####println(additional_clauses[end])

        # and, secondly, by adding (CNF[idx][1] || CNF[idx][2] || ... || CNF[idx][end]) -> y[idx],
        for l in 1:length(CNF[idx])
            push!(additional_clauses, [(-1) * CNF[idx][l], y_auxiliary_variables_disjunctions_substitutions[idx]])
            #####println("Adding for clauseidx=$idx the clause '(CNF[idx][1] || CNF[idx][2] || ... || CNF[idx][end]) -> y[idx])'")
            #####println(additional_clauses[end])
        end
    end
    # Add the constraint z <-> (y[1] && y[2] && ... && y[end]),
    # firstly, by adding z -> (y[1] && y[2] && ... && y[end]),
    for idx in 1:length(CNF)
        push!(additional_clauses, [(-1) * z_boolean_result_keeper, y_auxiliary_variables_disjunctions_substitutions[idx]])
        #####println("Adding the 'z -> (y[1] && y[2] && ... && y[end])' by the subsequent ANDs:")
        #####println(additional_clauses[end])
    end
    # and, secondly, by adding (y[1] && y[2] && ... && y[end]) -> Z
    push!(additional_clauses, vcat([(-1) * y_auxiliary_variables_disjunctions_substitutions[l] for l in 1:length(CNF)], [z_boolean_result_keeper]))
    #####println("Adding the clause '(y[1] && y[2] && ... && y[end]) -> Z'")
    #####println(additional_clauses[end])

    return additional_clauses
end

function runtest_symbolic_evaluation_of_cnf()
    examined_formula = [[-1, 2, 3], [2, -3]]
    num_clauses = length(examined_formula)
    last_sat_var_index = maximum(abs(el) for clause in examined_formula for el in clause)


    total_cnf_of_run = symbolic_evaluation_of_cnf(examined_formula, last_sat_var_index + 1, collect(last_sat_var_index+1+1:last_sat_var_index+1+num_clauses))
    @show(examined_formula)
    @show(total_cnf_of_run)

    total_cnf_of_run_pythonic = PyObject([PyObject([PyObject(literal) for literal in clause]) for clause in total_cnf_of_run])
    @pywith pysat.solvers.Solver(name="glucose421", bootstrap_with=total_cnf_of_run_pythonic, use_timer=true) as solver begin
        count_of_solutions = 0
        while solver.solve() #&& count_of_solutions < 5
            interpretation_found = solver.get_model()
            @show(interpretation_found)

            # For enumerating all solutions, now exclude the newly found one:
            solver.add_clause(PyObject([PyObject((-1) * el) for el in interpretation_found]))
            count_of_solutions += 1
            #break # currently, we are not interested in enumerating all solutions
        end
        @show(count_of_solutions)
    end
end

function permutation_matrix_constraints_in_cnf(mat)
    """
    Given a matrix whose entries are propositional literals x_{ij},
    model in the language of CNF that bi-stochasticity holds:
    sum_{j} x_{ij} = 1    :forall i
    sum_{i} x_{ij} = 1    :forall j

    Amount of clauses that will be used is quadratic in size(x)[1].
    """

    @assert size(mat)[1] == size(mat)[2] "Permutation matrices must be quadratic matrices; non-quadratic matrix detected!"

    additional_clauses = []

    for i in 1:size(mat)[1]
        # The `i'-th row must contain at least one `1':
        push!(additional_clauses, [mat[i, j] for j in 1:size(mat)[1]])

        # The `i'-th row can contain at most one `1':
        for cmb in Combinatorics.combinations(1:size(mat)[1], 2)
            push!(additional_clauses, [(-1) * mat[i, cmb[1]], (-1) * mat[i, cmb[2]]])
        end
    end

    for j in 1:size(mat)[1]
        # The `j'-th column must contain at least one `1':
        push!(additional_clauses, [mat[i, j] for i in 1:size(mat)[1]]) # TODO could be omittable right?

        # The `i'-th column can contain at most one `1': Can be omitted due to pigeonhole principle applying here
        #for cmb in Combinatorics.combinations(1:size(mat)[1], 2)
        #    push!(additional_clauses, [(-1) * mat[cmb[1], j], (-1) * mat[cmb[2], j]])
        #end
    end

    return additional_clauses

end

function permutation_to_permutation_matrix(perm)
    return [Int(j == perm[i]) for i in 1:length(perm), j in 1:length(perm)]
end

function runtests_permutation_matrix()
    total_cnf_of_run = []
    last_sat_var_index = 0
    n = 5
    plane_2d = [(i, j) for i in 1:n for j in 1:n]
    PI = [0 for i in 1:n, j in 1:n]

    dict_matrix = Dict()
    for (idx, el) in enumerate(plane_2d)
        PI[el[1], el[2]] = last_sat_var_index + idx
        dict_matrix[(el[1], el[2])] = last_sat_var_index + idx
    end
    last_sat_var_index += n * n

    append!(total_cnf_of_run, permutation_matrix_constraints_in_cnf(PI))

    total_cnf_of_run_pythonic = PyObject([PyObject([PyObject(literal) for literal in clause]) for clause in total_cnf_of_run])
    @pywith pysat.solvers.Solver(name="glucose3", bootstrap_with=total_cnf_of_run_pythonic, use_timer=true) as solver begin
        count_of_solutions = 0
        while solver.solve() #&& count_of_solutions < 5
            interpretation_found = solver.get_model()

            PI_reconstructed = [Int(Int(interpretation_found[dict_matrix[(i, j)]]) > 0) for i in 1:n, j in 1:n]
            @show(PI_reconstructed)

            # For enumerating all solution, now exclude the newly found one:
            solver.add_clause(PyObject([PyObject((-1) * el) for el in interpretation_found]))
            count_of_solutions += 1
            #break # currently, we are not interested in enumerating all solutions
        end
        @show(count_of_solutions)
        @show(count_of_solutions == factorial(n))
    end
end

function is_lex_descending(binary_string_first, binary_string_second)
    return (binary_string_first == binary_string_second || sortperm([binary_string_first, binary_string_second]) == [2, 1])
    # Check for equality added to cover the case of an underlying instable sort algorithm.
end

function apply_constant_permutation_to_symbolic_permutation_matrix(const_perm_gamma, permmat_PI)
    """
    Return the permutation matrix of gamma(pi(.)),
    where gamma is assumed to be passed as argument `const_perm_gamma'
    and pi is assumed to be passed in form of its respective permutation matrix `permmat_PI'.

    Under the permutation matrix of a permutation psi we understand the binary, bistochastic matrix PSI, guaranteeing
    that the matrix-vector product PSI*(1,2, ..., n) = (psi(1), psi(2), ..., psi(n)).

    Here an example:
    `
    julia> pi = [2,5,1,4,3]; PI = permutation_to_permutation_matrix(pi)
    julia> gamma = [3,2,1,5,4]; 
    julia> gamma_circ_pi = [gamma[pi[j]] for j in 1:length(pi)]
    julia> GAMMA_CIRC_PI = permutation_to_permutation_matrix(gamma_circ_pi)
    julia> GAMMA_CIRC_PI_accessed = apply_constant_permutation_to_symbolic_permutation_matrix_in_cnf(gamma, PI)
    julia> @show(GAMMA_CIRC_PI);
    julia> @show(GAMMA_CIRC_PI_accessed);
    julia> @show(GAMMA_CIRC_PI == GAMMA_CIRC_PI_accessed);
    '
    """
    inverse_of_const_perm_gamma = perminverting_function(const_perm_gamma)


    return [permmat_PI[i, inverse_of_const_perm_gamma[j]] for i in 1:length(const_perm_gamma), j in 1:length(const_perm_gamma)]
end

function symbolically_convert_permutation_matrix_to_incidence_matrix(PI, fresh_symbols_for_entries, fresh_symbols_for_comparison)
    """
    This function takes as input a symbolic permutation matrix `PI'
    (with assumed imposed binary and bistochastic constraints) and symbolically computes the value
    the incidence matrix X_pi of pi, where pi is the (unique) permutation associated to PI.
    The symbolic values are just stored for the strict upper triagonal part.
    Per each `cmb in Combinatorics.Combinations(n, 2)' (there are (n-1).n/2 many such subsets)
    we need (for first auxiliary set of variables)
    - 1 fresh variable representing the Boolean answer to the posed question of incidence.

    and (for second auxiliary set of variables)
    - n-1 fresh auxiliary variables for the appearing lexicographical comparison
    - note that the (n-1)-lexicographical-comparison-in-symbolic-fashion-with-answer needs
      a number of auxiliary variables corresponding to the number of its generated clauses,
      The number of generated clauses is here [[1 + ((n-1)-2)*5 + 4 + (n-1) == 6(n-1)-5 == 6n-11]].

    Managing the freshed variables has to be done hence in advance:

    last_sat_var_index = 777
    fresh_symbols_for_entries = []
    for (idx, cmb) in enumerate(Combinatorics.Combinations(n, 2))
        push!(fresh_symbols_for_entries, last_sat_var_index+idx)
    end
    last_sat_var_index += div(n*(n-1), 2)

    fresh_symbols_for_comparison = []
    for (idx, cmb) in enumerate(Combinatorics.Combinations(n, 2))
        a = [last_sat_var_index + i for i in 1:n-1]
        last_sat_var_index += n-1
        b = [last_sat_var_index + i for i in 1:(6*n-11)]
        last_sat_var_index += (6*n-11)
        push!(fresh_symbols_for_comparison, [a,b])
    end

    qqq = symbolically_convert_permutation_matrix_to_incidence_matrix(PI, fresh_symbols_for_entries, fresh_symbols_for_comparison)
    append!(total_cnf_of_run, qqq)
    """

    # Note that according to Lemma in paper, for i < j we have:
    # pi(i) < pi(j) iff (PI[i, l] : l = 1, ..., n) is a lexicographic successor of 
    # (PI[j, l] : l = 1, ..., n).

    println("Currently, we have:")
    @show(fresh_symbols_for_entries)
    @show(fresh_symbols_for_comparison)

    additional_clauses = []
    n = size(PI)[1]

    for (idx, cmb) in enumerate(Combinatorics.combinations(1:n, 2)) # i.e., Binomial([n], 2)

        row1 = [PI[cmb[1], j] for j in 1:n-1]
        row2 = [PI[cmb[2], j] for j in 1:n-1]
        println("=.==\nWe are adding the constraint that the row")
        @show(row2)
        println("is a lexicographic predecessor of")
        @show(row1)
        println("=.==\n")

        total_cnf_of_run_cmb = lex_and_ordering(row2, row1, fresh_symbols_for_comparison[idx][1])
        println("==== (this will be established by):")
        @show(total_cnf_of_run_cmb)
        println(", where the set of axuliary variables for lex-ordering was:")
        @show(fresh_symbols_for_comparison[idx][1])
        println("==== (endof this will be established by)")

        clauses_for_single_incidence = symbolic_evaluation_of_cnf(total_cnf_of_run_cmb, fresh_symbols_for_entries[idx], fresh_symbols_for_comparison[idx][2])
        println("Storing the truth value of lexicographical ordering is therefore:")
        @show(clauses_for_single_incidence)

        append!(additional_clauses, clauses_for_single_incidence)
    end

    return additional_clauses
end

function runtest_symbolically_convert_permutation_matrix_to_incidence_matrix()
    # As an intermediate task we want to try to find the permutation pi, such that
    # gamma(pi(.)) = [2, 4, 3, 5, 1], where gamma = [3, 2, 1, 5, 4]
    # Note: the solution to this task would be: pi == [2,5,1,4,3]

    gamma = [3, 2, 1, 5, 4]

    total_cnf_of_run = []
    last_sat_var_index = 0
    n = 3
    PI = [(i - 1) * n + j + last_sat_var_index for i in 1:n, j in 1:n]

    dict_matrix = Dict()
    for i in 1:n
        for j in 1:n
            dict_matrix[(i, j)] = (i - 1) * n + j + last_sat_var_index
        end
    end
    last_sat_var_index += n * n

    append!(total_cnf_of_run, permutation_matrix_constraints_in_cnf(PI))

    #PI_transformed = apply_constant_permutation_to_symbolic_permutation_matrix(gamma, PI)


    fresh_symbols_for_entries = []
    for idx in 1:binomial(n, 2) #(idx, cmb) in enumerate(Combinatorics.combinations(1:n, 2))
        push!(fresh_symbols_for_entries, last_sat_var_index + idx)
    end
    last_sat_var_index += binomial(n, 2)

    fresh_symbols_for_comparison = []
    for idx in 1:binomial(n, 2) #(idx, cmb) in enumerate(Combinatorics.combinations(1:n, 2))
        a = [last_sat_var_index + i for i in 1:n-1-1]
        last_sat_var_index += n - 1 - 1
        b = [last_sat_var_index + i for i in 1:(6*n-11-0-0-0-1)]
        last_sat_var_index += (6 * n - 11 - 0 - 0 - 0 - 1)
        push!(fresh_symbols_for_comparison, [a, b])
    end

    #qqq = symbolically_convert_permutation_matrix_to_incidence_matrix(PI_transformed, fresh_symbols_for_entries, fresh_symbols_for_comparison)
    #append!(total_cnf_of_run, qqq)
    qqq = symbolically_convert_permutation_matrix_to_incidence_matrix(PI, fresh_symbols_for_entries, fresh_symbols_for_comparison)
    append!(total_cnf_of_run, qqq)

    #for i in 1:n # impose one specific permutation as solution, dor debugging purposes:
    #    push!(total_cnf_of_run, [PI[i, i]])
    #end

    push!(total_cnf_of_run, [PI[1, 2]])
    push!(total_cnf_of_run, [PI[2, 3]])

    total_cnf_of_run_pythonic = PyObject([PyObject([PyObject(literal) for literal in clause]) for clause in total_cnf_of_run])
    @pywith pysat.solvers.Solver(name="glucose3", bootstrap_with=total_cnf_of_run_pythonic, use_timer=true) as solver begin
        count_of_solutions = 0
        while solver.solve() #&& count_of_solutions < 5
            interpretation_found = solver.get_model()

            PI_readout = [Int(Int(interpretation_found[dict_matrix[(i, j)]]) > 0) for i in 1:n, j in 1:n]
            @show(PI_readout)
            @show([Int(el) for el in interpretation_found])

            #PI_incidence_readout = [0 for i in 1:n, j in 1:n]
            #for (idx, cmb) in enumerate(Combinatorics.combinations(1:n, 2))
            #    PI_incidence_readout[cmb[1], cmb[2]] = 1
            #    PI_incidence_readout[cmb[2], cmb[1]] = 0
            #    #println("Interpretation has for pair $cmb the entry: $(interpretation_found[fresh_symbols_for_entries[idx]])")
            #end
            #@show(PI_incidence_readout)

            # For enumerating all solution, now exclude the newly found one:
            solver.add_clause(PyObject([PyObject((-1) * el) for el in interpretation_found]))
            count_of_solutions += 1
            #break # currently, we are not interested in enumerating all solutions
        end
        @show(count_of_solutions)
    end
end

function symbolically_compare_two_vectors_and_store_boolean_answer(vec1, vec2, comp_vars, m_asc, positive_literal)
    """
    Intermediate goal:
    give [0,1], [1,0] and save result true to fesh_symbol_for_result

    """

    additional_clauses = []

    tweak1 = lex_and_ordering(vcat([m_asc], vec1), vcat([positive_literal], vec2), comp_vars[1])
    tweak2 = lex_and_ordering(vcat([(-1) * positive_literal], [(-1) * literal for literal in vec1]), vcat([m_asc], [(-1) * literal for literal in vec2]), comp_vars[2])

    append!(additional_clauses, tweak1)
    append!(additional_clauses, tweak2)

    return additional_clauses
end

function experimentalfunc_symbolically_convert_permmat_to_incidence_mat(PI, PI_prime_incidence, last_sat_var_index_prime)

    n = size(PI)[1]


    total_cnf_of_run = []
    
    last_sat_var_index = last_sat_var_index_prime
    positive_literal = last_sat_var_index + 1
    last_sat_var_index += 1

    PI_incidence = Dict()

    for cmb in Combinatorics.combinations(1:n, 2)

        m_asc = PI_prime_incidence[(cmb[1], cmb[2])]

        vec1 = [PI[cmb[1], j] for j in 1:n]
        vec2 = [PI[cmb[2], j] for j in 1:n]


        comp_vars1 = collect(last_sat_var_index+1:last_sat_var_index+length(vec1)-1+1) # +1 for the prepend
        last_sat_var_index += length(vec1) - 1 + 1
        comp_vars2 = collect(last_sat_var_index+1:last_sat_var_index+length(vec1)-1+1) # +1 for the prepend
        last_sat_var_index += length(vec1) - 1 + 1
        comp_vars = [comp_vars1, comp_vars2]

        comp = symbolically_compare_two_vectors_and_store_boolean_answer(vec2, vec1, comp_vars, m_asc, positive_literal)
        append!(total_cnf_of_run, comp)

        println("Appended the following to total_cnf_of_run:")
        @show(comp)

        @show(m_asc)
    end
    @show(total_cnf_of_run)

    #=
    # Counting solutions modality:


    total_cnf_of_run_pythonic = PyObject([PyObject([PyObject(literal) for literal in clause]) for clause in total_cnf_of_run])


    @pywith pysat.solvers.Solver(name="glucose3", bootstrap_with=total_cnf_of_run_pythonic, use_timer=true) as solver begin
        count_of_solutions = 0
        while solver.solve() #&& count_of_solutions < 5
            interpretation_found = solver.get_model()

            solution = [Int(interpretation_found[idx]) for idx in 1:length(interpretation_found)]
            @show(solution)

            PI_incidence_recovered = [0 for i in 1:n, j in 1:n]
            for i in 1:n
                for j in i+1:n
                    PI_incidence_recovered[i, j] = Int(interpretation_found[PI_incidence[(i, j)]] > 0)
                    PI_incidence_recovered[j, i] = 1 - PI_incidence_recovered[i, j]
                end
            end
            @show(PI_incidence_recovered)
            # For enumerating all solution, now exclude the newly found one:
            solver.add_clause(PyObject([PyObject((-1) * el) for el in interpretation_found]))
            count_of_solutions += 1
            #break # currently, we are not interested in enumerating all solutions
        end
        @show(count_of_solutions)
    end
    =#

    return total_cnf_of_run
end

"""
The idea is to later TODO also permit permutation perm_assumptions passed via, e.g.:
"perm_assumptions" => [(1, [1, 2, 7, 4, 6, 3, 5]), (2, [1, 3, 5, 7, 6, 4, 2])], 
"""
function model_leftcoset(CONFIG)
    d = CONFIG["d"]
    n = CONFIG["n"]
    k = CONFIG["k"]
    kappa = CONFIG["kappa"]

    @assert kappa <= k "Interpolation parameter `kappa` may not exceed strength `k`!"

    if "subgroup_as_tuplelist" in keys(CONFIG)
        subgroup_as_tuplelist = CONFIG["subgroup_as_tuplelist"]
        @assert ("meant_for_leftcoset_approach" in keys(CONFIG)) "When a subgroup is passed, specifying the CONFIG is compulsory!"
        @assert d % length(subgroup_as_tuplelist) == 0 "The subgroup order must be a divisor of `d'!"
        @assert maximum(subgroup_as_tuplelist[1]) == n "Incompatibility of subgroup's alphabet size and `n'!"
    else
        @assert (!("meant_for_leftcoset_approach" in keys(CONFIG)) || CONFIG["meant_for_leftcoset_approach"] == true) "If no subgroup is specified, then the `meant_for_leftcoset_approach'-CONFIG may not be specified, or should at least be passed with value `true'!"
        subgroup_as_tuplelist = [collect(1:n)]
    end

    lex_strength = CONFIG["lex_strength"]

    @assert !("perm_assumptions" in keys(CONFIG)) "This is future work, please do not use in the current implementation."
    if "perm_assumptions" in keys(CONFIG)
        perm_assumptions = CONFIG["perm_assumptions"] # used to specify a certain number of offsets for the subgroup
    else
        perm_assumptions = []
    end

    @assert "card_constraint_mode" in keys(CONFIG) "Adapt to the newer implementation please!"

    if CONFIG["card_constraint_mode"] == "ILP"
        # TODO: Future work: do some modifications, where we essentially get everything except cardinality constraints from the CNF
        @assert false "Please do not fall back on this in the current development version!"
    end

    CNF_clauses = []
    XOR_clauses = []
    ILP_model = []

    PB_AT_MOST_CONSTRAINTS = []

    num_blocks = div(d, length(subgroup_as_tuplelist))
    access_tuples_indices_prime_z = [(l, i, j) for l in 1:num_blocks for i in 1:n for j in i+1:n]

    gamma = [el for el in subgroup_as_tuplelist]

    subpermutations = [subperm for subperm in Combinatorics.Permutations(1:n, kappa)] # Set of all S_{n,k}, e.g., S_{n,3}; in Paper [1], we have just mentioned the case kappa=3
    SOP = gen_SOP_with_cardinalities_range(kappa+1:k, n) # Set of SemiOrdered Patterns

    subpermutations_dtimes_wrapper = [tuple(l, el) for l in 1:d for el in subpermutations]
    SOP_dtimes_wrapper = [tuple(l, el) for l in 1:d for el in SOP]

    last_sat_var_index = 0
    r = Dict(value => key for (key, value) in enumerate(access_tuples_indices_prime_z))
    R = Dict([i < j ? ((l, i, j), r[(l, i, j)]) : (i == j ? ((l, i, i), nothing) : ((l, i, j), (-1) * r[(l, j, i)])) for l in 1:div(d, length(subgroup_as_tuplelist)) for i in 1:n for j in 1:n])

    last_sat_var_index = length(access_tuples_indices_prime_z)
    b = Dict(value => key + last_sat_var_index for (key, value) in enumerate(subpermutations_dtimes_wrapper))
    B = Dict([(item, b[item]) for item in subpermutations_dtimes_wrapper])

    last_sat_var_index += length(B)
    c = Dict(value => key + last_sat_var_index for (key, value) in enumerate(SOP_dtimes_wrapper))
    C = Dict([(item, c[item]) for item in SOP_dtimes_wrapper])
    last_sat_var_index += length(C)

    for idx_perm in perm_assumptions
        idx = idx_perm[1]
        perm_inverted = perminverting_function(idx_perm[2])
        for i in 1:n
            for j in i+1:n
                push!(CNF_clauses, [R[(idx, perm_inverted[i], perm_inverted[j])]])
            end
        end
    end

    # employ the following valid assumption in the non-heuristic setting, but we shrink ourselves to doing this per default only within the leftcoset-approach:
    if length(CONFIG["subgroup_as_tuplelist"]) == 1 && CONFIG["meant_for_leftcoset_approach"] == true # trivial subgroup, the second operand of and is satisfied anyways here but for the emphasis we keep this a marker!
        unit_clauses = []
        for i in 1:n
            for j in i+1:n
                push!(unit_clauses, [R[(1, i, j)]])
            end
        end
        append!(CNF_clauses, unit_clauses)
    end

    for l in 1:num_blocks
        for i in 1:n
            for j in i+1:n
                for h in [el for el in 1:n if el != i && el != j]
                    # R[(l, i, j)] & R[(l, j, h)] -> R[(l, i, h)] (transitivity):
                    # <=> -R[(l, i, j)] v -R[(l, j, h)] v R[(l, i, h)]
                    push!(CNF_clauses, [(-1) * R[(l, i, j)], (-1) * R[(l, j, h)], R[(l, i, h)]])
                end
            end
        end
    end

    # Add lexicographical ordering constraints, if accuracy is 0, no clauses will be added ... .
    if lex_strength > div(n * (n - 1), 2)
        # Sequence a(n), n=1, 2, ..., from OEIS-entry A036604, https://oeis.org/A036604 :
        a = [0, 1, 3, 5, 7, 10, 13, 16, 19, 22, 26, 30, 34, 38, 42, 46, 50, 54]
        # In contrast to a(n), we have the higher-values-sequence q(n) := n(n-1)/2:
        # q = [0, 1, 3, 6, 10, 15, 21, 28, 36, 45, 55, 66, 78, 91, 105, 120, 136, 153]

        lex_strength = div(n * (n - 1), 2)
        println("Automatically reduced `lex_strength` to its allowed maximum n*(n-1)/2=$(div(n*(n-1), 2)).")
        if n <= length(a)
            lex_strength = a[n] # TODO see other todo in document: not expressive enough, how can that be?????
            println("Overwrote `lex_strength` with its theoretically lowest-possible bound on number of (dynamic!) comparisons needed: $(a[n])")
        end
    end

    lex_order_clauses = []
    for l in 1:num_blocks-1
        index_traversal_l = []
        index_traversal_lplusone = []

        mat_prior = [R[(l, i, j)] for i in 1:n, j in 1:n]
        mat_successor = [R[(l + 1, i, j)] for i in 1:n, j in 1:n]

        index_traversal_l = traverse_sup_diagonals_linearly(mat_prior)[1:lex_strength]
        index_traversal_lplusone = traverse_sup_diagonals_linearly(mat_successor)[1:lex_strength]



        pool_of_new_vars = last_sat_var_index+1:last_sat_var_index+length(index_traversal_l)
        lexy = lex_and_ordering(index_traversal_lplusone, index_traversal_l, pool_of_new_vars)
        append!(lex_order_clauses, lexy)

        last_sat_var_index += length(index_traversal_l)
    end

    # @show lex_order_clauses

    append!(CNF_clauses, lex_order_clauses)

    # Linking constraints, binary incidence structure to pattern-coverage ("unfolded", with multiplicity)
    for l in 1:num_blocks
        for ll in 1:length(gamma)
            gamma_ll = gamma[ll]

            for subperm in subpermutations
                # Model that: B[((l-1)*len_block+ll, subperm)] -> R[(l, gamma_ll[subperm[j-1]], gamma_ll[subperm[j]])] , forall j=2,3, ..., kappa .

                for j in 2:length(subperm)
                    #gamma_ll_application = gamma_ll[pot_transpose(subperm[j-1], subperm[j])]
                    gamma_ll_application_prime = [gamma_ll[subperm[j-1]], gamma_ll[subperm[j]]]
                    gamma_ll_application = gamma_ll_application_prime




                    push!(CNF_clauses, [(-1) * B[((l - 1) * length(gamma) + ll, subperm)], R[(l, gamma_ll[subperm[j-1]], gamma_ll[subperm[j]])]]) # X^(theta_l) R[(l,...)]
                end

                # Model that: (\bigcap (R[(l, gamma_ll[subperm[jprime-1]], gamma_ll[semiseq[jprime]])] for jprime in 2:length(subperm)) ) -> B[((l-1)*length(gamma)+ll, subperm)]
                # i.e., \bigcup (- R[(l, gamma_ll[subperm[jprime-1]], gamma_ll[subperm[jprime]])] for jprime in 2:length(subperm) )  ;v;  B[((l-1)*length(gamma)+ll, subperm)]
                push!(
                    CNF_clauses, vcat(
                        [(-1) * R[(l,
                            #gamma_ll[pot_transpose(subperm[j-1], subperm[j])[1]],
                            #gamma_ll[pot_transpose(subperm[j-1], subperm[j])[2]])
                            gamma_ll[subperm[j-1]], gamma_ll[subperm[j]])
                        ]
                         for j in 2:length(subperm)
                        ],
                        [B[((l - 1) * length(gamma) + ll, subperm)]]
                    )
                )
            end
        end
    end
    for l in 1:num_blocks
        for ll in 1:length(gamma)
            gamma_ll = gamma[ll]

            for sop in SOP
                # Model that: C[((l-1)*len_block+ll, sop)] -> R[(l, gamma_ll[sop[1]], gamma_ll[sop[j]])] , forall j=2..
                for j in 2:length(sop)
                    push!(CNF_clauses, [(-1) * C[((l - 1) * length(gamma) + ll, sop)], R[(l, gamma_ll[sop[1]], gamma_ll[sop[j]])]])
                end

                # Model that: (\bigcap (R[(l, gamma_ll[sop[1]], gamma_ll[semiseq[jprime]])] for jprime in 2:length(sop)) ) -> C[((l-1)*length(gamma)+ll, sop)]
                # i.e., \bigvee (- R[(l, gamma_ll[sop[1]], gamma_ll[sop[jprime]])] for jprime in 2:length(sop) )  ;v;  C[((l-1)*length(gamma)+ll, sop)]
                push!(
                    CNF_clauses, vcat(
                        [(-1) * R[(l, gamma_ll[sop[1]], gamma_ll[sop[jprime]])] for jprime in 2:length(sop)],
                        [C[((l - 1) * length(gamma) + ll, sop)]]
                    )
                )
            end
        end
    end # ... endof linking constraints

    debugpurpose_counter_of_card_constr = 0

    # Cardinality Constraints here, first for B-coverage:
    for subperm in subpermutations

        card_constraint_cuurent_literals = PyObject([PyObject(B[(l, subperm)]) for l in 1:d])

        # We use pysat.pb.PBEnc.atmost from see https://pysathq.github.io/docs/html/api/pb.html

        push!(PB_AT_MOST_CONSTRAINTS, [card_constraint_cuurent_literals, div(d, factorial(length(subperm)))])

        card_constraint = pysat.pb.PBEnc.atmost(lits=card_constraint_cuurent_literals, bound=div(d, factorial(length(subperm))), encoding=pysat.pb.EncType.best, top_id=last_sat_var_index).clauses


        if typeof(card_constraint) == Matrix{Int64}
            # attention this, sometimes, returns a matrix instead of a list of lists:
            card_constraint_repairer = [card_constraint[i, :] for i in 1:size(card_constraint)[1]]
            card_constraint = copy(card_constraint_repairer)
        end
        fresh_vars = collect(setdiff(Set([(2 * Int(el > 0) - 1) * el for cl in card_constraint for el in cl]), Set([(2 * Int(el > 0) - 1) * el for cl in card_constraint_cuurent_literals for el in cl])))

        # :::::::::::::Module details:::::::::::::::::
        # class pysat.pb.EncType
        # This class represents a C-like enum type for choosing the pseudo-Boolean encoding to use. The values denoting
        # the encodings are:
        # best = 0
        # bdd = 1
        # seqcounter = 2
        # sortnetwrk = 3
        # adder = 4
        # binmerge = 5
        # The desired encoding can be selected either directly by its integer identifier, e.g. 2, or by its alphabetical name,
        # e.g. EncType.seqcounter.
        # All the encodings are produced and returned as a list of clauses in the pysat.formula.CNF format.
        # Note that the encoding type can be set to best, in which case the encoder selects one of the other encodings
        # from the list (in most cases, this invokes the bdd encoder)

        if CONFIG["card_constraint_mode"] == "pysatCNF"
            append!(CNF_clauses, copy(card_constraint))
        end
        debugpurpose_counter_of_card_constr += 1

        if !isempty(fresh_vars)
            last_sat_var_index = maximum(fresh_vars)
        end
    end

    for subperm in subpermutations
        # Add here the respective XOR clause:
        push!(XOR_clauses, [[B[(l, subperm)] for l in 1:d-1]; ((div(d, factorial(length(subperm))) % 2 == 1) ? 1 : -1) * B[(d, subperm)]])
    end

    # Cardinality constraints here, now for C-coverage
    length(SOP) == 0 && println("SOP is empty.")

    for sop in SOP
        card_constraint_cuurent_literals = PyObject([PyObject(C[(l, sop)]) for l in 1:d])

        push!(PB_AT_MOST_CONSTRAINTS, [card_constraint_cuurent_literals, div(d, length(sop))])
        card_constraint = pysat.pb.PBEnc.atmost(lits=card_constraint_cuurent_literals, bound=div(d, length(sop)), encoding=pysat.pb.EncType.best, top_id=last_sat_var_index).clauses

        if typeof(card_constraint) == Matrix{Int64}
            # Attention this, sometimes, returns a matrix instead of a list of lists; here is the fix:
            card_constraint_repairer = [card_constraint[i, :] for i in 1:size(card_constraint)[1]]
            card_constraint = copy(card_constraint_repairer)
        end
        fresh_vars = collect(setdiff(Set([(2 * Int(el > 0) - 1) * el for cl in card_constraint for el in cl]), Set([(2 * Int(el > 0) - 1) * el for cl in card_constraint_cuurent_literals for el in cl])))
        if CONFIG["card_constraint_mode"] == "pysatCNF"
            append!(CNF_clauses, copy(card_constraint))
        end
        debugpurpose_counter_of_card_constr += 1

        if !isempty(fresh_vars)
            last_sat_var_index = maximum(fresh_vars)
        end
    end

    # Cardinality Constraints here, now also for C-coverage:
    for sop in SOP
        # Add here the respective XOR clause:
        push!(XOR_clauses, [[C[(l, sop)] for l in 1:d-1]; ((div(d, length(sop)) % 2 == 1) ? 1 : -1) * C[(d, sop)]])
    end

    return [CNF_clauses, PB_AT_MOST_CONSTRAINTS, XOR_clauses, ILP_model, R, B, C]
end

function model_rightcoset(CONFIG)
    d = CONFIG["d"]
    n = CONFIG["n"]
    k = CONFIG["k"]
    kappa = CONFIG["kappa"]

    @assert kappa <= k "Interpolation parameter `kappa` may not exceed strength `k`!"

    if "subgroup_as_tuplelist" in keys(CONFIG)
        subgroup_as_tuplelist = CONFIG["subgroup_as_tuplelist"]
        @assert ("meant_for_leftcoset_approach" in keys(CONFIG)) "When a subgroup is passed, specifying the CONFIG is compulsory!"
        @assert d % length(subgroup_as_tuplelist) == 0 "The subgroup order must be a divisor of `d`!"
        @assert maximum(subgroup_as_tuplelist[1]) == n "Incompatibility of subgroup's alphabet size and `n`!"
    else
        @assert (!("meant_for_leftcoset_approach" in keys(CONFIG)) || CONFIG["meant_for_leftcoset_approach"] == true) "If no subgroup is specified, then the `meant_for_leftcoset_approach`-CONFIG may not be specified, or should at least be passed with value `true`!"
        subgroup_as_tuplelist = [collect(1:n)]
    end

    lex_strength = CONFIG["lex_strength"]

    if "perm_assumptions" in keys(CONFIG)
        perm_assumptions = CONFIG["perm_assumptions"] # used to specify a certain number of offsets for the subgroup
    else
        perm_assumptions = []
    end

    CNF_clauses = []
    XOR_clauses = []
    ILP_model = [] # TODO future work

    PB_AT_MOST_CONSTRAINTS = []

    last_sat_var_index = 0

    PI_LIST = [] # List of offsets each stored as permutation matrix
    PI_PRIME_INCIDENCES_LIST = [] # List of offsets each stored as incidence matrix 

    positive_literal = last_sat_var_index + 1
    last_sat_var_index += 1

    gamma = [el for el in subgroup_as_tuplelist]
    num_blocks = div(d, length(gamma))
    access_tuples_indices_prime_z = [(l, i, j) for l in 1:num_blocks for i in 1:n for j in i+1:n]


    for block_idx in 1:num_blocks
        PI = [last_sat_var_index + (i - 1) * n + j for i in 1:n, j in 1:n]
        push!(PI_LIST, PI)
        append!(CNF_clauses, permutation_matrix_constraints_in_cnf(PI))

        last_sat_var_index += n * n
        for gamma_ll in gamma
            PI_prime = apply_constant_permutation_to_symbolic_permutation_matrix(gamma_ll, PI)
            PI_prime_incidence = Dict()

            for i in 1:n
                for j in i+1:n
                    PI_prime_incidence[(i, j)] = last_sat_var_index + 1
                    last_sat_var_index += 1
                end
            end

            push!(PI_PRIME_INCIDENCES_LIST, PI_prime_incidence)

            for cmb in Combinatorics.combinations(1:n, 2)

                m_asc = PI_prime_incidence[(cmb[1], cmb[2])]

                vec1 = [PI_prime[cmb[1], j] for j in 1:n]
                vec2 = [PI_prime[cmb[2], j] for j in 1:n]


                comp_vars1 = collect(last_sat_var_index+1:last_sat_var_index+length(vec1)-1+1) # +1 for the prepend
                last_sat_var_index += length(vec1) - 1 + 1
                comp_vars2 = collect(last_sat_var_index+1:last_sat_var_index+length(vec1)-1+1) # +1 for the prepend
                last_sat_var_index += length(vec1) - 1 + 1
                comp_vars = [comp_vars1, comp_vars2]

                comp = symbolically_compare_two_vectors_and_store_boolean_answer(vec2, vec1, comp_vars, m_asc, positive_literal)
                #println("Appended the following to `CNF_clauses`:")
                #@show(comp)
                append!(CNF_clauses, comp)

                #@show(m_asc)
            end
        end
    end

    subpermutations = [subperm for subperm in Combinatorics.Permutations(1:n, kappa)] # Set of all S_{n,k}, e.g., S_{n,3}; in Paper [1], we have just mentioned the case kappa=3
    SOP = gen_SOP_with_cardinalities_range(kappa+1:k, n) # Set of SemiOrdered Patterns

    subpermutations_dtimes_wrapper = [tuple(l, el) for l in 1:d for el in subpermutations]
    SOP_dtimes_wrapper = [tuple(l, el) for l in 1:d for el in SOP]


    #r = Dict(value => key for (key, value) in enumerate(access_tuples_indices_prime_z))
    #R = Dict([i < j ? ((l, i, j), r[(l, i, j)]) : (i == j ? ((l, i, i), nothing) : ((l, i, j), (-1) * r[(l, j, i)])) for l in 1:div(d, length(subgroup_as_tuplelist)) for i in 1:n for j in 1:n])
    # last_sat_var_index +=

    ##### last_sat_var_index = length(access_tuples_indices_prime_z)
    b = Dict(value => key + last_sat_var_index for (key, value) in enumerate(subpermutations_dtimes_wrapper))
    B = Dict([(item, b[item]) for item in subpermutations_dtimes_wrapper])

    #println("We introduced the linking variables to subpermutations B given by:")
    #@show(B)

    last_sat_var_index += length(B)
    c = Dict(value => key + last_sat_var_index for (key, value) in enumerate(SOP_dtimes_wrapper))
    C = Dict([(item, c[item]) for item in SOP_dtimes_wrapper])
    last_sat_var_index += length(C)

    # in fact, we do not need transitivity, but could impose it ... because perhaps a bit more efficient than the permutation_matrix encoding
    add_aswell_transitivity = false
    if add_aswell_transitivity == true
        for l in 1:d
            for i in 1:n
                for j in i+1:n
                    for h in [el for el in 1:n if el != i && el != j]
                        # (Transitivity, the form for R)
                        #     R[(l, i, j)] & R[(l, j, h)]  -> R[(l, i, h)] 
                        # <=> -R[(l, i, j)] v -R[(l, j, h)] v R[(l, i, h)]
                        push!(CNF_clauses,
                            [
                                (-1) * PI_PRIME_INCIDENCES_LIST[l][(i, j)],
                                (-1) * (j < h ? PI_PRIME_INCIDENCES_LIST[l][(j, h)] : (-1) * PI_PRIME_INCIDENCES_LIST[l][(h, j)]),
                                i < h ? PI_PRIME_INCIDENCES_LIST[l][(i, h)] : (-1) * PI_PRIME_INCIDENCES_LIST[l][(h, i)]
                            ]
                        )
                    end
                end
            end
        end
    end

    # Linking constraints, binary incidence structure to pattern-coverage ("unfolded", with multiplicity)
    for l in 1:num_blocks
        for ll in 1:length(gamma)
            for subperm in subpermutations
                # Model that: B[((l-1)*len_block+ll, subperm)] -> R[(l, gamma_ll[subperm[j-1]], gamma_ll[subperm[j]])] , forall j=2,3, ..., kappa .
                tmp_clauses = []
                for j in 2:length(subperm)
                    push!(tmp_clauses,
                        [
                            (-1) * B[((l - 1) * length(gamma) + ll, subperm)],
                            subperm[j-1] < subperm[j] ? PI_PRIME_INCIDENCES_LIST[(l-1)*length(gamma)+ll][subperm[j-1], subperm[j]] : (-1) * PI_PRIME_INCIDENCES_LIST[(l-1)*length(gamma)+ll][subperm[j], subperm[j-1]]
                        ]
                    ) # X^(theta_l) R[(l,...)]
                end
                append!(CNF_clauses, tmp_clauses)

                # Model that: (\bigwedge (R[(l, gamma_ll[subperm[jprime-1]], gamma_ll[semiseq[jprime]])] for jprime in 2:length(subperm)) ) -> B[((l-1)*length(gamma)+ll, subperm)]
                # i.e., \bigvee(- R[(l, gamma_ll[subperm[jprime-1]], gamma_ll[subperm[jprime]])] for jprime in 2:length(subperm) )  ;v;  B[((l-1)*length(gamma)+ll, subperm)]
                tmp_clauses2 = []
                push!(
                    tmp_clauses2, vcat(
                        [(-1) * (subperm[j-1] < subperm[j] ? PI_PRIME_INCIDENCES_LIST[(l-1)*length(gamma)+ll][(subperm[j-1], subperm[j])] : (-1) * PI_PRIME_INCIDENCES_LIST[(l-1)*length(gamma)+ll][(subperm[j], subperm[j-1])])
                         for j in 2:length(subperm)
                        ],
                        [B[((l - 1) * length(gamma) + ll, subperm)]]
                    )
                )
                append!(CNF_clauses, tmp_clauses2)

                #println("In order to cover $subperm , we added the link:")
                #@show(tmp_clauses)
                #@show(tmp_clauses2)
            end
        end
    end

    debugpurpose_counter_of_card_constr = 0

    # Cardinality Constraints here, first for B-coverage:
    for subperm in subpermutations

        card_constraint_cuurent_literals = PyObject([PyObject(B[(l, subperm)]) for l in 1:d])

        # We use pysat.pb.PBEnc.atmost from see https://pysathq.github.io/docs/html/api/pb.html

        ### push!(PB_AT_MOST_CONSTRAINTS, [card_constraint_cuurent_literals, div(d, factorial(length(subperm)))])

        card_constraint = pysat.pb.PBEnc.atmost(lits=card_constraint_cuurent_literals, bound=div(d, factorial(length(subperm))), encoding=pysat.pb.EncType.best, top_id=last_sat_var_index).clauses
        if typeof(card_constraint) == Matrix{Int64}
            # attention this, sometimes, returns a matrix instead of a list of lists;
            # here is the fix:
            card_constraint_repairer = [card_constraint[i, :] for i in 1:size(card_constraint)[1]]
            card_constraint = copy(card_constraint_repairer)
        end
        fresh_vars = collect(setdiff(Set([(2 * Int(el > 0) - 1) * el for cl in card_constraint for el in cl]), Set([(2 * Int(el > 0) - 1) * el for cl in card_constraint_cuurent_literals for el in cl])))

        # :::::::::::::Module details:::::::::::::::::
        # class pysat.pb.EncType
        # This class represents a C-like enum type for choosing the pseudo-Boolean encoding to use. The values denoting
        # the encodings are:
        # best = 0
        # bdd = 1
        # seqcounter = 2
        # sortnetwrk = 3
        # adder = 4
        # binmerge = 5
        # The desired encoding can be selected either directly by its integer identifier, e.g. 2, or by its alphabetical name,
        # e.g. EncType.seqcounter.
        # All the encodings are produced and returned as a list of clauses in the pysat.formula.CNF format.
        # Note that the encoding type can be set to best, in which case the encoder selects one of the other encodings
        # from the list (in most cases, this invokes the bdd encoder)

        if CONFIG["card_constraint_mode"] == "pysatCNF"
            append!(CNF_clauses, copy(card_constraint))
            #println("Appended coverage-among-rows constraint:")
            #@show(card_constraint)
        end
        debugpurpose_counter_of_card_constr += 1

        if !isempty(fresh_vars)
            last_sat_var_index = maximum(fresh_vars)
        end
    end

    # Add lexicographical ordering constraints, if accuracy is 0, no clauses will be added ... .
    if lex_strength > div(n * (n - 1), 2)
        # Sequence a(n), n=1, 2, ..., from OEIS-entry A036604, https://oeis.org/A036604 :
        a = [0, 1, 3, 5, 7, 10, 13, 16, 19, 22, 26, 30, 34, 38, 42, 46, 50, 54]
        # In contrast to a(n), we have the higher-values-sequence q(n) := n(n-1)/2:
        # q = [0, 1, 3, 6, 10, 15, 21, 28, 36, 45, 55, 66, 78, 91, 105, 120, 136, 153]

        lex_strength = div(n * (n - 1), 2)
        println("Automatically reduced `lex_strength` to its allowed maximum n*(n-1)/2=$(div(n*(n-1), 2)).")
        if n <= length(a)
            lex_strength = a[n] # TODO see other todo in document: not expressive enough, how can that be?????
            println("Overwrote `lex_strength` with its theoretically lowest-possible bound on number of comparisons needed: $(a[n])")
        end
    end

    lex_order_clauses = []
    for l in 1:num_blocks-1
        index_traversal_l = []
        index_traversal_lplusone = []

        mat_prior = [i < j ? PI_PRIME_INCIDENCES_LIST[l][(i, j)] : (i > j ? (-1) * PI_PRIME_INCIDENCES_LIST[l][(j, i)] : 0) for i in 1:n, j in 1:n]
        mat_successor = [i < j ? PI_PRIME_INCIDENCES_LIST[l+1][(i, j)] : (i > j ? (-1) * PI_PRIME_INCIDENCES_LIST[l+1][(j, i)] : 0) for i in 1:n, j in 1:n]

        index_traversal_l = traverse_sup_diagonals_linearly(mat_prior)[1:lex_strength]
        index_traversal_lplusone = traverse_sup_diagonals_linearly(mat_successor)[1:lex_strength]

        pool_of_new_vars = last_sat_var_index+1:last_sat_var_index+length(index_traversal_l)
        lexy = lex_and_ordering(index_traversal_lplusone, index_traversal_l, pool_of_new_vars)
        append!(lex_order_clauses, lexy)

        # @show index_traversal_l
        # @show index_traversal_lplusone
        # @show pool_of_new_vars
        # @show lexy

        last_sat_var_index += length(index_traversal_l)
    end
    # @show lex_order_clauses

    append!(CNF_clauses, lex_order_clauses)

    # Add first-permutation-is-identity-Assumption legit for the right-coset approach:
    unit_clauses = []
    for i in size(PI_LIST[1])[1]
        for j in size(PI_LIST[1])[1]
            push!(unit_clauses, [(i == j ? 1 : -1) * PI_LIST[1][i, j]])
        end
    end
    append!(CNF_clauses, unit_clauses)

    #println("Showing entire formula inclusively coverage:")
    #@show(CNF_clauses)

    return [CNF_clauses, PB_AT_MOST_CONSTRAINTS, XOR_clauses, ILP_model, PI_LIST, PI_PRIME_INCIDENCES_LIST, B, C]
end

function solvemodel_leftcoset(CONFIG)
    # CONFIG["solver_employed"] = "glucose3"
    # CONFIG["solver_employed"] = "cryptominisat" # [https://pysathq.github.io/docs/html/api/solvers.html, Finally, PySAT offers rudimentary support of CryptoMiniSat5 [3] through the interface provided by the `pycryptosat' package]
    # CONFIG["solver_employed"] = "cadical195"
    # first install cryptominisat, e.g., see [https://github.com/msoos/cryptominisat, Section "Compiling in Linux"]


    # -> pip install pycryptosat

    # See also https://pysathq.github.io/docs/html/api/solvers.html :

    # pysat.solvers.SolverNames
    #
    # This class serves to determine the solver requested by a user given a string name. This allows for using several possible names for specifying a solver.
    #
    # cadical103  = ('cd', 'cd103', 'cdl', 'cdl103', 'cadical103')
    # cadical153  = ('cd15', 'cd153', 'cdl15', 'cdl153', 'cadical153') # TODO has native at most constraints
    # cadical195  = ('cd19', 'cd195', 'cdl19', 'cdl195', 'cadical195')
    # cryptosat   = ('cms', 'cms5', 'crypto', 'crypto5', 'cryptominisat', 'cryptominisat5')
    # gluecard3   = ('gc3', 'gc30', 'gluecard3', 'gluecard30')
    # gluecard41  = ('gc4', 'gc41', 'gluecard4', 'gluecard41')
    # glucose3    = ('g3', 'g30', 'glucose3', 'glucose30')
    # glucose4    = ('g4', 'g41', 'glucose4', 'glucose41')
    # glucose42   = ('g42', 'g421', 'glucose42', 'glucose421')
    # lingeling   = ('lgl', 'lingeling')
    # maplechrono = ('mcb', 'chrono', 'maplechrono')
    # maplecm     = ('mcm', 'maplecm')
    # maplesat    = ('mpl', 'maple', 'maplesat')
    # mergesat3   = ('mg3', 'mgs3', 'mergesat3', 'mergesat30')
    # minicard    = ('mc', 'mcard', 'minicard')
    # minisat22   = ('m22', 'msat22', 'minisat22')
    # minisatgh   = ('mgh', 'msat-gh', 'minisat-gh')


    CNF_clauses, pb_card_constr, xor_clauses, ilp, R, B, C = model_leftcoset(CONFIG)

    valueR = Dict([i < j ? ((l, i, j), 0) : (i == j ? ((l, i, i), 0) : ((l, i, j), 0)) for l in 1:CONFIG["d"] for i in 1:CONFIG["n"] for j in 1:CONFIG["n"]])
    interpretation_found = nothing # for SAT
    is_satisfiable = false

    CNF_clauses_as_py_obj = PyObject([PyObject([PyObject(el) for el in row]) for row in CNF_clauses])


    if CONFIG["solver_employed"] == "cryptominisat" && CONFIG["card_constraint_mode"] == "XORrelaxation"

        cnf2print = pysat.formula.CNF(from_clauses=CNF_clauses_as_py_obj)
        cnf2print.to_file("some-file-name.cnf")

        # Quickfix (interactive experimental mode):
        run(`sed -i '1d' some-file-name.cnf`)

        open("some-file-name.cnf", "a") do o::IO
            for el in xor_clauses
                # Append XOR-clause to .cnf file:
                println(o, "x$(join(el, " ")) 0")
            end
        end
        println("XOR clauses now written to cnf file ....")

        println("Now starting the command.")
        some_command = "~/cryptominisat/build/cryptominisat5 some-file-name.cnf > ~/solverresult.out"
        p = subprocess.Popen(some_command, stdout=subprocess.PIPE, shell=true)
        (output, err) = p.communicate()
        #This makes the wait possible
        p_status = p.wait()

        return nothing # interactive experimental mode
    end

    solver_finished_in_time = false
    
    # startof preparation for proof logging
    cnf2print = pysat.formula.CNF(from_clauses=CNF_clauses_as_py_obj)
    timestamp = Dates.format(now(), "yyyyMMdd_HHmmss")
    cnf2print.to_file("export_d$(CONFIG["d"])_n$(CONFIG["n"])_k$(CONFIG["k"])_kappa$(CONFIG["kappa"])_sgo$(length(CONFIG["subgroup_as_tuplelist"]))_timestamp$(timestamp).cnf")
    # endof preparation for proof logging

    @pywith pysat.solvers.Solver(name=CONFIG["solver_employed"], bootstrap_with=CNF_clauses_as_py_obj, use_timer=true) as m begin
        if CONFIG["solver_employed"] == "cadical195" && CONFIG["card_constraint_mode"] == "native_card_constraints"
            m.activate_atmost() # activate native at most constraints, works only for Cadical195
            for constr in pb_card_constr
                m.add_atmost(lits=PyObject([PyObject(el) for el in constr[1]]), k=PyObject(constr[2]), no_return=false)
            end
        end

        py"""
        was_interrupted = False
        def interrupt(my_s):
            global was_interrupted
            was_interrupted = True
            my_s.interrupt()

        """
        encapsulated_pyobj_fnc_interrupt = py"interrupt" # if this does not have its own line we get some error (?)
        encapsulated_pyobj_timer = py_threading.Timer(CONFIG["timelimit_given"], encapsulated_pyobj_fnc_interrupt, [m])
        encapsulated_pyobj_timer.start()

        julias_tic_measured = @CPUelapsed begin
            println("SAT solver $(CONFIG["solver_employed"]) started now on leftcoset approach.")
            is_satisfiable = m.solve_limited(expect_interrupt=PyObject(true))
        end
        forced_interruption_happened = py"was_interrupted" # get this crucial information on solver termination p_status
        solver_finished_in_time = !forced_interruption_happened

        if solver_finished_in_time
            encapsulated_pyobj_timer.cancel() # otherwise, the process will wait until the entire residual time (meant for the SAT solver) is consumed.
        end


        if forced_interruption_happened == true
            is_satisfiable = false # just by convention set to `false`; indeed, we cannot know.
        end

        solvertime_measured = m.time()
        @show(is_satisfiable)
        @show(julias_tic_measured) # up to small differences this measurement agrees with the internal one of satsolver in subsequent row
        @show(solvertime_measured)
        @show(m.accum_stats())
        @show(solver_finished_in_time)

        if is_satisfiable
            println("Has solution!")
            interpretation_found = m.get_model()


            for l in 1:div(CONFIG["d"], length(CONFIG["subgroup_as_tuplelist"]))
                for i in 1:CONFIG["n"]
                    for j in i+1:CONFIG["n"]
                        #println("value(R[($l, $i, $j)]) is $(value(R[(l, i, j)]))")
                        valueR[(l, i, j)] = Int(interpretation_found[R[(l, i, j)]] > 0)
                        valueR[(l, j, i)] = 1 - valueR[(l, i, j)]
                    end
                end
            end

            offset_permutations = []
            pdesign_obtained = []
            supdiagonal_pad = []

            for l in 1:div(CONFIG["d"], length(CONFIG["subgroup_as_tuplelist"]))
                pgenerated = [0 for j in 1:CONFIG["n"]]
                for i in 1:CONFIG["n"]
                    pgenerated[i] = CONFIG["n"] - sum(valueR[(l, i, j)] for j in 1:CONFIG["n"])
                end

                push!(offset_permutations, pgenerated)

                for ll in 1:length(CONFIG["subgroup_as_tuplelist"])
                    push!(pdesign_obtained, apply_perm(pgenerated, CONFIG["subgroup_as_tuplelist"][ll]))
                end
            end

            println("recall, group is:\n", as_prettyfied_string_permutationslist(CONFIG["subgroup_as_tuplelist"], false), "\n")
            println("pdesign_obtained is:\n", as_prettyfied_string_permutationslist(pdesign_obtained, false), "\n")

            for l in 1:div(CONFIG["d"], length(CONFIG["subgroup_as_tuplelist"]))
                push!(supdiagonal_pad, traverse_sup_diagonals_linearly(permutation_to_incidencematrix(offset_permutations[l])))
            end

            println("offsets list is:\n[")
            for (idx, el) in enumerate(offset_permutations)
                println(el, ",   diagpad: ", supdiagonal_pad[idx])
            end
            println("]\n")


            println("The sanity check for $(CONFIG["kappa"])-rankwise-coverage on $(CONFIG["n"]) symbols and $(CONFIG["d"]) rows tells us: $(is_rkw_indep(pdesign_obtained, CONFIG["kappa"])) \n========endof rkw-sanitycheck")
            println("The sanity check for $(CONFIG["k"])-restr. minwise-coverage on $(CONFIG["n"]) symbols and $(CONFIG["d"]) rows tells us: $(is_mw_indep(pdesign_obtained, CONFIG["k"])) \n========endof mw-sanitycheck")


            query = [is_lex_descending(supdiagonal_pad[i], supdiagonal_pad[i+1]) for i in 1:length(supdiagonal_pad)-1]
            sanity_check_descendingness = all(query)
            println("The sanity check for descending lex-order tells us: $sanity_check_descendingness : $query")

            m_accumstats_saving = "$(m.accum_stats())"
            m.delete() # cleaning-up
            return [offset_permutations, pdesign_obtained, is_mw_indep(pdesign_obtained, CONFIG["k"]) && is_rkw_indep(pdesign_obtained, CONFIG["kappa"]), solver_finished_in_time, solvertime_measured, julias_tic_measured, m_accumstats_saving]
        else
            println(solver_finished_in_time ? "UNSAT CNF INSTANCE!" : "ANSWER FROM SAT SOLVER INCONCLUSIVE DUE TO TIME-OVERFLOW")

            #println("Do you want to get the unsatisfiable core? (y/yc) ... yes vs yes-completely")
            #user_input = readline()
            #CORE = nothing
            #if user_input == "y"
            #    CORE = m.get_core()
            #    println("Core length is $(length(CORE))")
            #elseif user_input == "yc"
            #    CORE = m.get_core()
            #    println("Core is $CORE")
            #end
            m_accumstats_saving = "$(m.accum_stats())"
            m.delete() # cleaning-up
            return [[], [], false, solver_finished_in_time, solvertime_measured, julias_tic_measured, m_accumstats_saving]
        end
    end # endof @pywith
end

function solvemodel_rightcoset(CONFIG)
    CNF_clauses, pb_card_constr, xor_clauses, ilp, T_permmatrices_propositionalvariable_indices, T_incidencematrices_propositionalvariable_indices, B, C = model_rightcoset(CONFIG)
    CNF_clauses_as_py_obj = PyObject([PyObject([PyObject(literal) for literal in clause]) for clause in CNF_clauses])


    offset_vec_recovered = []
    pdesign_obtained = []
    supdiagonal_pad = []

    @pywith pysat.solvers.Solver(name=CONFIG["solver_employed"], bootstrap_with=CNF_clauses_as_py_obj, use_timer=true) as m begin
        if CONFIG["solver_employed"] == "cadical195" && CONFIG["card_constraint_mode"] == "native_card_constraints"
            m.activate_atmost() # TODO future work: activate native at most constraints, works only for Cadical195
            for constr in pb_card_constr
                m.add_atmost(lits=PyObject([PyObject(el) for el in constr[1]]), k=PyObject(constr[2]), no_return=false)
            end
        end

        py"""
        was_interrupted = False
        def interrupt(my_s):
            global was_interrupted
            was_interrupted = True
            my_s.interrupt()

        """
        encapsulated_pyobj_fnc_interrupt = py"interrupt" # if this does not have its own line we get some error (?)
        encapsulated_pyobj_timer = py_threading.Timer(CONFIG["timelimit_given"], encapsulated_pyobj_fnc_interrupt, [m])
        encapsulated_pyobj_timer.start()

        julias_tic_measured = @CPUelapsed begin
            println("SAT solver $(CONFIG["solver_employed"]) started now on rightcoset approach.")
            is_satisfiable = m.solve_limited(expect_interrupt=PyObject(true))
        end
        forced_interruption_happened = py"was_interrupted" # get this crucial information on solver termination p_status
        solver_finished_in_time = !forced_interruption_happened

        if solver_finished_in_time
            encapsulated_pyobj_timer.cancel() # otherwise the process will wait until the entire residual time (meant for the SAT solver) is consumed.
        end


        if forced_interruption_happened == true
            is_satisfiable = false # just by convention set to `false`; indeed, we cannot know.
        end

        solvertime_measured = m.time()
        @show(is_satisfiable)
        @show(julias_tic_measured) # up to small differences this measurement agrees with the internal one of satsolver in subsequent row
        @show(solvertime_measured)
        @show(m.accum_stats())
        @show(solver_finished_in_time)


        if is_satisfiable
            interpretation_found = m.get_model()

            #solution = [Int(interpretation_found[idx]) for idx in 1:length(interpretation_found)]
            #@show(solution)

            theta_all_incidencematrices_recovered = [[0 for i in 1:CONFIG["n"], j in 1:CONFIG["n"]] for l in 1:CONFIG["d"]]
            for l in 1:CONFIG["d"]
                for i in 1:CONFIG["n"]
                    for j in i+1:CONFIG["n"]
                        theta_all_incidencematrices_recovered[l][i, j] = Int(interpretation_found[T_incidencematrices_propositionalvariable_indices[l][(i, j)]] > 0)
                        theta_all_incidencematrices_recovered[l][j, i] = 1 - theta_all_incidencematrices_recovered[l][i, j]
                    end
                end
            end

            T_permmatrices_recovered = [[0 for i in 1:CONFIG["n"], j in 1:CONFIG["n"]] for l in 1:div(CONFIG["d"], length(CONFIG["subgroup_as_tuplelist"]))]
            for l in 1:div(CONFIG["d"], length(CONFIG["subgroup_as_tuplelist"]))
                for i in 1:CONFIG["n"]
                    for j in 1:CONFIG["n"]
                        T_permmatrices_recovered[l][i, j] = Int(interpretation_found[T_permmatrices_propositionalvariable_indices[l][i, j]] > 0)
                    end
                end
            end

            #@show(theta_all_incidencematrices_recovered)
            #@show(T_permmatrices_recovered)
            offset_vec_recovered = [permutationmatrix_to_permutation(el) for el in T_permmatrices_recovered]
            #@show(offset_vec_recovered)

            for l in 1:div(CONFIG["d"], length(CONFIG["subgroup_as_tuplelist"]))
                for ll in 1:length(CONFIG["subgroup_as_tuplelist"])
                    push!(pdesign_obtained, apply_perm(CONFIG["subgroup_as_tuplelist"][ll], offset_vec_recovered[l]))
                end
            end

            ####################################################
            println("recall, group is:\n", as_prettyfied_string_permutationslist(CONFIG["subgroup_as_tuplelist"], false), "\n")
            println("pdesign_obtained is:\n", as_prettyfied_string_permutationslist(pdesign_obtained, false), "\n")


            for l in 1:div(CONFIG["d"], length(CONFIG["subgroup_as_tuplelist"]))
                push!(supdiagonal_pad, traverse_sup_diagonals_linearly(permutation_to_incidencematrix(offset_vec_recovered[l])))
            end

            println("offsets list is:\n[")
            for (idx, el) in enumerate(offset_vec_recovered)
                println(el, ",   diagpad: ", supdiagonal_pad[idx])
            end
            println("]\n")

            println("The sanity check for $(CONFIG["kappa"])-rankwise-coverage on $(CONFIG["n"]) symbols and $(CONFIG["d"]) rows tells us: $(is_rkw_indep(pdesign_obtained, CONFIG["kappa"])) \n========endof rkw-sanitycheck")
            println("The sanity check for $(CONFIG["k"])-restr. minwise-coverage on $(CONFIG["n"]) symbols and $(CONFIG["d"]) rows tells us: $(is_mw_indep(pdesign_obtained, CONFIG["k"])) \n========endof mw-sanitycheck")


            query = [is_lex_descending(supdiagonal_pad[i], supdiagonal_pad[i+1]) for i in 1:length(supdiagonal_pad)-1]
            sanity_check_descendingness = all(query)
            println("The sanity check for descending lex-order tells us: $sanity_check_descendingness : $query")
            ##############################################

            m_accumstats_saving = "$(m.accum_stats())"
            m.delete() # cleaning-up
            return [offset_vec_recovered, pdesign_obtained, is_mw_indep(pdesign_obtained, CONFIG["k"]) && is_rkw_indep(pdesign_obtained, CONFIG["kappa"]), solver_finished_in_time, solvertime_measured, julias_tic_measured, m_accumstats_saving]
        else
            println(solver_finished_in_time ? "UNSAT CNF INSTANCE!" : "ANSWER FROM SAT SOLVER INCONCLUSIVE DUE TO TIME-OVERFLOW")

            m_accumstats_saving = "$(m.accum_stats())"
            m.delete() # cleaning-up
            return [[], [], false, solver_finished_in_time, solvertime_measured, julias_tic_measured, m_accumstats_saving]
        end
    end
end


function carryout_search(CONFIG)
    if CONFIG["meant_for_leftcoset_approach"] == true
        # offset_vec_recovered, pdesign_obtained, correctnes, solver_finished_in_time, tic_measured, accum_stats = 
        return solvemodel_leftcoset(CONFIG)
    else # i.e., we are in the rightcoset approach:
        #offset_vec_recovered, pdesign_obtained, correctnes, solver_finished_in_time, tic_measured, accum_stats =
        return solvemodel_rightcoset(CONFIG)
    end
end

function left_trivialrun_for_debugging()

    demo = 1
    if demo == 1 # delivers a non-existence proof
        subgroups_examined = [
            [[1, 2, 3, 4, 5, 6]]
        ]
        alphabet_size = 6
        d = 12
        k = 4
        kappa = 3 # (recall, always 3 for mw-indep)
        lexstrength = div(6*5,2)-1
        #CMD = "~/pysat/solvers/glucose421/glucose-4.2.1/build/glucose-simp export_d12_n6_k4_kappa3_sgo1_timestamp20254403_0002216.cnf -vbyte -certified -certified-output=proof.out"
    	#CMD2 = "~/drat-trim-master/drat-trim export_d12_n6_k4_kappa3_sgo1_timestamp20254403_0002216.cnf proof.out"
    elseif demo == 2 # delivers a non-existence proof
        d = 24
        k = 4
        kappa = 3
        alphabet_size = 8
        subgroups_examined = [
            [[1, 2, 3, 4, 5, 6, 7, 8]]
        ]
        lexstrength = div(8*7,2)-1
    end

    for subgroup in subgroups_examined
        CONFIG = Dict(
            "meant_for_leftcoset_approach" => true,
            "d" => d,
            "n" => alphabet_size,
            "k" => k,
            "kappa" => kappa,
            "subgroup_as_tuplelist" => subgroup,
            "card_constraint_mode" => "pysatCNF",
            "lex_strength" => lexstrength,
            "solver_employed" => "glucose3",
            "timelimit_given" => 3600
        )
        carryout_search(CONFIG)
    end
end

function right_trivialrun_for_debugging()

    demo = 1
    if demo==1
        # see how long the 9,3,3 approach needs for a 2-subgroup-rightcoset:
        subgroups_examined = [[[1, 2, 3, 4, 5, 6, 7, 8, 9], [1, 2, 3, 9, 8, 7, 6, 5, 4]]]
        alphabet_size = 9
        d = 18
        k = 3
        kappa = 3
    end

    for subgroup in subgroups_examined
        CONFIG = Dict(
            "meant_for_leftcoset_approach" => false,
            "d" => d,
            "n" => alphabet_size,
            "k" => k,
            "kappa" => kappa,
            "subgroup_as_tuplelist" => subgroup,
            "card_constraint_mode" => "pysatCNF",
            "lex_strength" => 13,
            "solver_employed" => "glucose3"
        )
        carryout_search(CONFIG)
    end
end

TARGET_DB_PATH = "~/experiments_collection.sqlite"
steer_experim_viaSQL(TARGET_DB_PATH, "SELECT * FROM constellations WHERE meant_for_leftcoset_approach=1", 10^10, true)
#steer_experim_viaSQL("~/experiments_collection.sqlite", "SELECT * FROM constellations WHERE meant_for_leftcoset_approach=0 AND d=12 AND n = 4 AND k = 4 AND kappa = 3 and (subgrouporder=12 OR subgrouporder=6)", 1, true)

# Interesting isolated constellations:
#left_trivialrun_for_debugging()
#right_trivialrun_for_debugging()
